<?php

namespace App\Http\Controllers;

use Exception;
use App\Models\User;
use Illuminate\Http\Request;
use App\Http\Requests\UserRequest;
use Spatie\Permission\Models\Role;
use Illuminate\Support\Facades\Hash;
use App\Http\Requests\ProfileUpdateRequest;
use App\Http\Requests\UserUpdateRequest;

class ProfileController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        if ($request->has('search')) {
            $keyword = $request->input('search');
            $paginate = false;
            $users = User::where('name', 'LIKE', '%' . $keyword . '%')
                ->orWhere('email', 'LIKE', '%' . $keyword . '%')
                ->orWhere('status', 'LIKE', '%' . $keyword . '%')
                ->orderBy('id', 'DESC')->with('roles')->get();

            return view('user.list', compact('users', 'paginate'));
        } else {
            $paginate = true;
            $users = User::orderBy('id', 'DESC')->with('roles')->paginate(10)->onEachSide(0);
            return view('user.list', compact('users', 'paginate'));
        }
    }

    /**
     * Display the user's profile form.
     */
    public function edit(User $user)
    {
        $roles = Role::get();
        $user = $user->with('roles')->find($user->id);
        return view('user.edit', compact('user', 'roles'));
    }
 
     /**
     * Display the user's profile form.
     */
    public function profile(User $user)
    {
        $roles = Role::get();
        $user = $user->with('roles')->find($user->id);
        return view('user.profile', compact('user', 'roles'));
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(UserRequest $request)
    {
        try {
            $profile_picture = $this->upload_image($request, "");

            $user = new User();
            $user->name = $request->input('name');
            $user->email = $request->input('email');
            $user->status = $request->input('status');
            $user->password = Hash::make($request->input('password'));
            $user->profile_picture = $profile_picture;

            $user->save();

            $role = $request->input('role');
            $user->assignRole($role);

            return redirect()->route('user.create')->with('success', 'Added successfully');
        } catch (Exception $e) {
            return back()->with('error', 'Please enter a unique email address');
        }
    }

    /**
     * Update the user profile
     */
    public function update_profile(ProfileUpdateRequest $request, User $user) {
        try {
            $profile_picture = $this->upload_image($request, $user->profile_picture);

            $user->name = $request->input('name');
            $user->email = $request->input('email');
            $user->password = Hash::make($request->input('password'));
            $user->profile_picture = $profile_picture;
            $user->save();
 
            return back()->with('success', 'Updated successfully');
        } catch (Exception $e) {
            return back()->with('error', 'Please enter a unique email address');
        }
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        $roles = Role::get();
        return view('user.create', compact('roles'));
    }

    /**
     * Update the user's profile information.
     */
    public function update(UserUpdateRequest $request, User $user)
    {
        try {
            $profile_picture = $this->upload_image($request, $user->profile_picture);

            $user->name = $request->input('name');
            $user->email = $request->input('email');
            $user->status = $request->input('status');
            $user->password = Hash::make($request->input('password'));
            $user->profile_picture = $profile_picture;
            $user->save();

            $role = $request->input('role');
            $user->syncRoles($role);

            return redirect()->route('user.index')->with('success', 'Updated successfully');
        } catch (Exception $e) {
            return back()->with('error', 'Please enter a unique email address');
        }
    }

    /**
     * Delete the user's account.
     */
    public function destroy(User $user)
    {
        $user->syncRoles([]);
        $user->delete();

        return redirect()->route('user.index')->with('success', 'Deleted successfully');
    }

    // UPLOAD THE NEW IMAGE
    private function upload_image(Request $request, $name)
    {
        if ($request->hasfile('profile_picture')) {
            if (!empty($name)) {
                $this->remove_image($name);
            }
            $file = $request->file('profile_picture');
            $name = unique_image_name($file->getClientOriginalExtension());
            $file->move('images/profile', $name);
        }

        return $name;
    }

    // REMOVE THE OLD IMAGE FROM THE STORAGE
    private function remove_image($name)
    {
        if (!empty($name)) {
            $path = 'images/profile/';
            $file_old = $path . $name;
            if (file_exists($file_old)) {
                unlink($file_old);
            }
        }
    }
}